# Python interface for NetCDF  - M. Neish 2008/09/21
# - Quick 'n dirty
# - Read-only access
# - You must already know what you're looking for
# - Values are read as a 1D array, since Python doesn't handle higher dimensions
#   that well.
#
# Examples:
#
# Import the module:
#   import ncquick
# Open a file:
#   file = ncquick.open ("blah.nc")
# Get time variable:
#   time = file.var("time")
# Print units of time:
#   print time.att("units")
# Get all values of time:
#   print time.values()
# Get the dimensions of v:
#   v = file.var("v")
#   print v.dims    // i.e ["time", "lat", "lon"]
#   print v.size    // i.e [24, 180, 360]
# (note that the values are retrieved in a 1D array, so it's up to you to
#  reshape them using whatever numerical package is available to you.)
# Close the file
#   ncquick.close (file)


from ctypes import *

libnetcdf = CDLL("libnetcdf.so")

# Some codes from the netcdf library
NC_GLOBAL = -1

NC_BYTE = 1
NC_CHAR = 2
NC_SHORT = 3
NC_INT = 4
NC_LONG = NC_INT
NC_FLOAT = 5
NC_DOUBLE = 6

NC_MAX_NAME = 256

class NCError (Exception):
  def __init__ (self, value): self.value = value
  def __str__ (self): return self.value

class Var:
  def __init__ (self, fileid, name, id) :
    self.fileid = fileid
    self.name = name
    self.id = id
    # Get additional info
    type = c_int()
    libnetcdf.nc_inq_vartype(fileid, id, byref(type))
    ndims = c_int()
    libnetcdf.nc_inq_varndims(fileid, id, byref(ndims));
    dimids = (c_int*ndims.value)()
    libnetcdf.nc_inq_vardimid (fileid, id, dimids);
    self.ndims = ndims.value
    self.type = type.value
    self.dims = []
    self.size = []
    for d in dimids:
      len = c_int()
      libnetcdf.nc_inq_dimlen(fileid, d, byref(len))
      self.size.append(len.value)
      name = create_string_buffer(NC_MAX_NAME+1)
      libnetcdf.nc_inq_dimname(fileid, d, name)
      self.dims.append(name.value)
    self.length = 1
    for s in self.size: self.length *= s

  def att (self, attribute) :
    len = c_int()
    type = c_int()
    err = libnetcdf.nc_inq_att (self.fileid, self.id, c_char_p(attribute), 
                                byref(type), byref(len))
    if err != 0: raise NCError("can't find attribute '"+attribute+"' in var '"+self.name+"'")
    if type.value != NC_CHAR: raise NCError ("can only handle string attributes in this version")
    string = create_string_buffer(len.value)
    err = libnetcdf.nc_get_att_text (self.fileid, self.id,
                                     c_char_p(attribute), string)
    if err != 0: raise NCError
    return string.value

  def values (self) :
    len = self.length
    if self.type == NC_BYTE:
      data = (c_byte*len)()
      err = libnetcdf.nc_get_var_ubyte(self.fileid, self.id, data)
    elif self.type == NC_SHORT:
      data = (c_short*len)()
      err = libnetcdf.nc_get_var_short(self.fileid, self.id, data)
    elif self.type == NC_INT:
      data = (c_int*len)()
      err = libnetcdf.nc_get_var_int(self.fileid, self.id, data)
    elif self.type == NC_LONG:
      data = (c_long*len)()
      err = libnetcdf.nc_get_var_long(self.fileid, self.id, data)
    elif self.type == NC_FLOAT:
      data = (c_float*len)()
      err = libnetcdf.nc_get_var_float(self.fileid, self.id, data)
    elif self.type == NC_DOUBLE:
      data = (c_double*len)()
      err = libnetcdf.nc_get_var_double(self.fileid, self.id, data)
    else: raise NCError("unknown data type")
    if err != 0: raise NCError("data read error")
    return data[:]

class File:
  def __init__ (self, name, id) :
    self.name = name
    self.id = id
    self.glob = Var (id, "global", NC_GLOBAL)

  def var (self, varname) :
    varid = c_int()
    err = libnetcdf.nc_inq_varid (self.id, c_char_p(varname), byref(varid))
    if err != 0: raise NCError("can't file variable '"+varname+"' in file '"+self.name+"'")
    return Var (self.id, varname, varid)

def open (filename) :
  fileid = c_int()
  err = libnetcdf.nc_open (c_char_p(filename), 0, byref(fileid))
  if err != 0: raise NCError("can't open file '"+filename+"'")
  return File (filename, fileid)

def close (file) :
  err = libnetcdf.nc_close (file.id)
  if err != 0: raise NCError("can't close file '"+file.name+"'")



