load "./reformat/coord_function.ncl"

; ******************************************************************
; Calculate long term monthly means (monthly climatology)
; requires named dimensions
;
; The time dimension must be a multiple of 12
;
;    x(time,lat)  <==== INPUT DIMENSION ORDER
;    x!0 = "time"
;    x!1 = "lat"
;
undef("clmMonTL")
function clmMonTL (x[*][*]:numeric)
local dimx, nlat, ntim, i, nmo, nmos, xLT, aveMonth
begin
  dimx = dimsizes(x)
  ntim = dimx(0)
  nlat = dimx(1)

  nmos  = 12
  modCheck ("clmMonTL", ntim, nmos)   ; error check

  do i=0,1
    if (ismissing(x!i)) then
      print("calc_seasonal.ncl: clmMonTL: all dimensions must be named")
      exit
    end if
  end do

  xLT = x($x!1$|:,$x!0$|:)

;
; Compute all 12 monthly averages first.
;
  aveMonth = new((/nlat,nmos/),typeof(x),getFillValue(x))

  do nmo=0,nmos-1
    aveMonth(:,nmo) = dim_avg( xLT(:,nmo:ntim-1:nmos) )
  end do

; copy attributes and add a couple of informational attributes
; only need to do this if plotting or writing to netCDF file

  copy_VarAtts (x,aveMonth)
  aveMonth@time_op_ncl = "Climatology: "+ (ntim/nmos) +" years"

; copy spatial (eg, lat) coordinate variables

  aveMonth!0 = xLT!0
  if (iscoord(xLT,xLT!0)) then
    aveMonth&$aveMonth!0$ = xLT&$xLT!0$
  end if

  aveMonth!1     = "month"           ; create a "month" named dim
  aveMonth&month = ispan(0,nmos-1,1) ; create a month coord var

  return ( aveMonth(month|:,$x!1$|:) )   ; return (month,lat)
end

; ******************************************************************
undef("calc_SEASON")
function calc_SEASON(data,sss)
local data_new, arrS, n, ds, rank, dimNames, data_sss, data_mod, idx_mm
begin
  ; data: 2D (time,lat)
  ; data: 3D (time,lat,lon)
  ; data: 4D (time,lev,lat,lon)

  arrS = (/"ANN","DJF","JJA","MAM","SON"/)
  n     = ind(arrS.eq.sss)  ; index corresponding to season
  if (ismissing(n)) then
      print ("calc_SEASON: bad season: SEASON="+sss)
      exit
  end if

  ds = dimsizes(data)
  rank = dimsizes(ds)

  if (ds(0) .gt. 12) then
    if (rank .eq. 2) then
      data_mod = clmMonTL(data)
    end if
    if (rank .eq. 3) then
      data_mod = clmMonTLL(data)
    end if
    if (rank .eq. 4) then
      data_mod = clmMonTLLL(data)
    end if
  else
    data_mod = data
  end if

  if (rank .eq. 2) then
    data_new = new((/ds(1)/),typeof(data))
  end if
  if (rank .eq. 3) then
    data_new = new((/ds(1),ds(2)/),typeof(data))
  end if
  if (rank .eq. 4) then
    data_new = new((/ds(1),ds(2),ds(3)/),typeof(data))
  end if
  copy_VarAtts(data,data_new)
  dimNames = getvardims(data)

  do i=1,rank-1,1
    ip1 = i-1
    data_new!ip1 = dimNames(i)
    data_new&$dimNames(i)$=data&$dimNames(i)$
  end do

  if (sss .eq. "ANN") then
    data_sss = month_to_annual(data_mod,1)
  else
    if (sss .eq. "DJF") then
      idx_mm = (/0,1,11/)
    end if
    if (sss .eq. "JJA") then
      idx_mm = (/5,6,7/)
    end if
    if (sss .eq. "MAM") then
      idx_mm = (/2,3,4/)
    end if
    if (sss .eq. "SON") then
      idx_mm = (/8,9,10/)
    end if

    if (rank .eq. 2) then
      data_sss = (data_mod(idx_mm(0),:)+data_mod(idx_mm(1),:)+data_mod(idx_mm(2),:))/3.0
    end if
    if (rank .eq. 3) then
      data_sss = (data_mod(idx_mm(0),:,:)+data_mod(idx_mm(1),:,:)+data_mod(idx_mm(2),:,:))/3.0
    end if
    if (rank .eq. 4) then
      data_sss = (data_mod(idx_mm(0),:,:,:)+data_mod(idx_mm(1),:,:,:)+data_mod(idx_mm(2),:,:,:))/3.0
    end if
  end if

  data_new =(/data_sss/)
  copy_VarAtts(data,data_new)

  return(data_new)

end

; ******************************************************************
undef("day_to_month")
function day_to_month(data,fno)
local data_new, t_new, tim, dt, ds, t_utc, nt, t_st_yr, t_ed_yr, dimNames, \
      year, month, idx_i, idx_ym, rank
begin
  ; data: T2Ds(time,lat,lon), T2Dz(time,lev,lat), T1Ds(time,lat)

  ds = dimsizes(data)
  rank = dimsizes(ds)

  tim = data&time
  tim = coord_list("time",tim)
  dt = dimsizes(tim)

  t_utc = ut_calendar(tim,0)
  t_st_yr = floattoint(t_utc(0,0))
  t_ed_yr = floattoint(t_utc(dimsizes(tim)-1,0))
  nt = (t_ed_yr - t_st_yr + 1) * 12

  if (rank .eq. 2) then
    data_new = new((/nt,ds(1)/),typeof(data))
  end if
  if (rank .eq. 3) then
    data_new = new((/nt,ds(1),ds(2)/),typeof(data))
  end if

  copy_VarAtts(data,data_new)
  dimNames = getvardims(data)
  t_new = new((/nt/),typeof(tim))
  t_new@units = tim@units

  do i=1,rank-1
    data_new!i = dimNames(i)
    data_new&$dimNames(i)$=data&$dimNames(i)$
  end do

  do year=t_st_yr,t_ed_yr
    do month=1,12
      idx_i = (year-t_st_yr)*12+(month-1)
      idx_ym = ind((t_utc(:,0).eq.year).and.(t_utc(:,1).eq.month))
      if (.not.all(ismissing(idx_ym))) then
        if (rank .eq. 3) then
	  if (fno .eq. "T2Ds") then
	    data_new(idx_i,:,:) = (/dim_avg_Wrap(data(lat|:,lon|:,time|min(idx_ym):max(idx_ym)))/)
          else
            data_new(idx_i,:,:) = (/dim_avg_Wrap(data(plev|:,lat|:,time|min(idx_ym):max(idx_ym)))/)
          end if
        else
          data_new(idx_i,:) = (/dim_avg_Wrap(data(lat|:,time|min(idx_ym):max(idx_ym)))/)
        end if
        t_new(idx_i) = avg(tim(min(idx_ym):max(idx_ym)))
      else
        t_new(idx_i) = (/ut_inv_calendar(year,month,15,0,0,0,tim@units,0)/)
      end if
      delete(idx_ym)
    end do
  end do

  data_new!0 = "time"
  data_new&time = t_new

  return(data_new)

end

undef("extract_time_range")
function extract_time_range(data,startYr,endYr)
local data_new, tim, t_utc, t_utc_yr, idx_t
begin
  ; data: T's

  ds = dimsizes(data)
  rank = dimsizes(ds)

  tim = data&time
;WHAT Is this for? Probably not necessary.
;  tim = coord_list("time",tim)
  
  t_utc = ut_calendar(tim,0)
  t_utc_yr = floattoint(t_utc(:,0))
  idx_t = ind((t_utc_yr.ge.startYr).and.(t_utc_yr.le.endYr))

  if (rank .eq. 2) then
    data_new = data(idx_t,:)
  end if
  if (rank .eq. 3) then
    data_new = data(idx_t,:,:)
  end if
  if (rank .eq. 4) then
    data_new = data(idx_t,:,:,:)
  end if

  return(data_new)

end
