;*********************************************************************
;This is the main ncl program called from the script file cf_convert.py
;
;contact  I. Cionni     i.cionni@niwa.co.nz
;
; Changes
; 2009-02-25   CAF  add hyam,hybm,hyai,hybi,p0 to output file if it's in the input file
; 2009-06-11   CAF  added date range to output filename
; 2009-06-12   CAF  fixed dimension problem with T2Dz
; 2010-09-15   HS/SWW   fixed lev->plev through out file, changed input file, fil_in
;                       and general tidy-up including comments
; --------------------------------------------------------------------
; attribute.ncl
; 2010-11-30   HS   handle multiple input files and read data in blocks with limited size (maxDataSize).
;                   data gets fixed if project_name & model specified file exists or if its field and rank do not match.
;                   output a file with only selected time range, without "Define Mode".
; 2011-01-28   HS   bug fix - an error when start_date and end_date are same.
; 2011-02-02   ZL   change type of variables (start_date & end_date).
; 2011-08-26   HS   bug fix - an error on selecting proper time range.
; 2012-06-07   HS   change the location of fix_file (function fix_data).
;
;*********************************************************************

load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"

;*********************************************************************
;check that all these functions are in the same directory of attribute.ncl
;*********************************************************************
load "convert2flt.ncl"
load "time_conform.ncl"
load "coord_function.ncl"
load "att_function.ncl"
load "global_attribute.ncl"

begin
   ; if ProjectName_ModelName.ncl exists, then call it.
   MODEL        = getenv("MODEL")
   project_name = getenv("PROJECT_NAME")

   fix_file = "./fix_"+project_name+"/"+project_name+"_"+MODEL+".ncl"
   data_to_fix = isfilepresent(fix_file)
   if data_to_fix then
      loadscript(fix_file)
   end if
end

begin

   fields_list = (/"T2Ms: Monthly-mean 2-d atmosphere or land surface data (longitude, latitude, time:month)",\
                   "T3M: Monthly-mean 3-d atmosphere data (longitude, latitude, pressure, time:month) ",\
                   "T2Mz: Monthly-mean zonal mean 2-d atmosphere or land surface data (longitude, pressure, time:month)",\
                   "T1Ms: Monthly-mean 1-d atmosphere or land surface data on a certain pressure level (latitude, time:month)",\
                   "T2Ds: Daily-mean 2-d atmosphere data (longitude, latitude, time:day)",\
                   "T3D: Daily-mean 3-d atmosphere data (longitude, latitude, pressure, time:day)",\
                   "T2Dz: Daily-mean zonal mean 2-d atmosphere  data (latitude, pressure, time:month)",\
                   "T2Is: Daily instantaneous 2-d atmosphere data for all years (longitude, latitude, time:day)",\
                   "T3I: Daily-instantaneous 3-d atmosphere data for selected years (longitude, latitude, model levels, time:day)",\
                   "T2Iz: Daily instantaneous zonal mean 2-d atmosphere data for all years (latitude, pressure, time:day)",\
                   "T1Iz: Daily instantaneous 1-d field for all years (latitude-pressure, time:day)",\
                   "T0I: Daily instantaneous 0-d field for all years (time:day)",\
                   "T0As: Annual-mean 0-d atmosphere or land surface data on a certain pressure level (latitude, time:year)",\
                   "F2Ms: 2D Time indipendent land surface data Latitude-Longitude",\
                   "TO2Ms: Monthly-mean 2-d ocean or sea ice data (longitude, latitude, time:month)"/)

;===========================================================
; get environment variables
;===========================================================
   dir        = getenv ("inputdir")
   inputfile  = getenv ("inputfile")
;   fil_in = dir+"/"+inputfile
   dur        = getenv ("outputdir")
   fol        = getenv ("outputfile")
   var_name   = getenv ("var_name")
   source     = getenv ("source")
   contact    = getenv ("contact")
   comment    = getenv("comment")
   REF        = getenv ("REF")
   exp_id     = getenv ("exp_id")
   history    = getenv ("history")
   references = getenv ("references")
   field      = getenv ("CCMVal_field")
   start_date = stringtoint(getenv("START_DATE"))
   end_date   = stringtoint(getenv("END_DATE"))
   maxDataBlock = stringtofloat(getenv("MAX_DATA_BLOCKSIZE"))

   if (.not.isvar("time_name")) then
      time_name = getenv ("time_name")
   end if
   if (.not.isvar("lat_name")) then
      lat_name = getenv ("lat_name")
   end if
   if (.not.isvar("lon_name")) then
      lon_name = getenv ("lon_name")
   end if
   if (.not.isvar("lev_name")) then
      lev_name = getenv ("lev_name")
   end if

; create output filename and delete if it exists (only occurs for the force_processing option)
; this allows cf_convert.py to return 1 (=error) if attribute.ncl doesn't finish properly.
   ;fol= project_name+"_"+REF+"_"+MODEL+"_"+exp_id+"_"+field+"_"+var_name+"_"+start_date+"-"+end_date+".nc"
   fn_out = dur+fol
   if (isfilepresent(fn_out)) then
      system("/bin/rm " +fn_out )
   end if

; check field of input-file
   att_field = str_match(fields_list,field)
   if (ismissing(att_field)) then
      print("[attribute.ncl] " + field + " is wrong CCMVal field name: please check your setting!")
      exit
   end if

;===========================================================
; input & output files
;===========================================================
; get input-file(s)
   fils = systemfunc ("ls "+dir+"/"+inputfile) ; file paths

   if(all(ismissing(fils))) then
      print ("[attribute.ncl] Files not found. "+ dir+"/"+inputfile)
      exit
   end if

; create output file
   fout = addfile(fn_out,"c")
   ; make time and UNLIMITED dimension
   filedimdef(fout,"time",-1,True)

   first_write = True
   file_pointer = 0

;===========================================================
; Main loop - go through file(s)
;===========================================================
   do i=0,dimsizes(fils)-1

      fin = addfile (fils(i), "r")

      vNames = getfilevarnames(fin)

      ; special case t850
      if (var_name.eq."t850") then
         var_name = "ta"
      end if

      if (sum(where(vNames.eq.var_name,1,0)).eq.0) then
         print ("[attribute.ncl] FATAL: var="+var_name+" is not in the file")
         exit
      end if

      dNames = getfilevardims(fin,var_name)
      vards = getfilevardimsizes(fin,var_name)
      rank = dimsizes(vards)
      varType = getfilevartypes(fin,var_name)

      ; calculate variable data size
      ; datasize = systemfunc("ls -s "+fils(i)+" | cut -d ' ' -f1")
      if (varType.eq."double") then
         datasize=8.
      else
         datasize=4.
      end if

      do j=0,rank-1,1
         datasize = datasize * vards(j)
      end do

      datasize = datasize/1000000  ; in Mbyte
      ;print("datasize="+datasize)

      ; set time start/end indices for reading blocks
      time_len = vards(ind(dNames.eq.time_name))
      if (datasize .gt. maxDataBlock) then
         nblocks = floattoint(ceil(datasize / maxDataBlock))
         time_block = time_len/nblocks
         if (mod(time_len,nblocks).ne.0) then
            time_block = time_block+1
         end if
         i_st = new(nblocks,"integer")
         i_ed = i_st
         do j=0,nblocks-1
            i_st(j) = j*time_block
            i_ed(j) = (j+1)*time_block-1
         end do
         i_ed(nblocks-1) = time_len-1
      else
         i_st = 0
         i_ed = time_len-1
         nblocks = 1
      end if
      delete(dNames)
      delete(time_len)
      delete(vards)

      ; read in blocks
      do j=0,nblocks-1

         ; get a block of data
         if (rank.eq.4) then
            varData0 = fin->$var_name$($time_name$|i_st(j):i_ed(j),$lev_name$|:,$lat_name$|:,$lon_name$|:)
         end if
         if (rank.eq.3) then
            if isStrSubset(field,"s") then
               varData0 = fin->$var_name$($time_name$|i_st(j):i_ed(j),$lat_name$|:,$lon_name$|:)
            else
               varData0 = fin->$var_name$($time_name$|i_st(j):i_ed(j),$lev_name$|:,$lat_name$|:)
            end if
         end if
         if (rank.eq.2) then
            varData0 = fin->$var_name$($time_name$|i_st(j):i_ed(j),$lat_name$|:)
         end if
         if (rank.eq.1) then
            varData0 = fin->$var_name$($time_name$|i_st(j):i_ed(j))
         end if

         ; if project_name_model.ncl exists, then call fix_data.
         if data_to_fix then
            if isdefined("fix_data") then
               temp = fix_data(varData0)
               delete(varData0)
               varData0 = temp
               delete(temp)
            end if
         end if

         ; select time dimension and check for range
         tim = varData0&$time_name$
         time0 = time_conform(tim)
         tim_ut = ut_calendar(time0,4)
         ind_t = ind(tim_ut.ge.start_date.and.tim_ut.lt.(end_date+1))
         delete(tim)
         delete(time0)
         delete(tim_ut)

         if (.not.all(ismissing(ind_t)))

            if (rank.eq.4) then
               varData1 = varData0(ind_t,:,:,:)
            end if
            if (rank.eq.3) then
               varData1 = varData0(ind_t,:,:)
            end if
            if (rank.eq.2) then
               varData1 = varData0(ind_t,:)
            end if
            if (rank.eq.1) then
               varData1 = varData0(ind_t)
            end if
            delete(varData0)

            ; double2float, convert2flt
            if (varType.eq."double") then
               temp = dble2flt(varData1)
               delete(varData1)
               varData1 = temp
               delete(temp)
            end if

            varData = (/varData1/)
            copy_VarAtts(varData1,varData)

            ; dimensions: check name, type and attributes
            dNames = getvardims(varData1)
            do k=0,rank-1,1
               varDimX = varData1&$dNames(k)$
               ; dimensions also need to be float
               if (.not.isfloat(varDimX)) then
                  temp = convert2flt(varDimX)
                  delete(varDimX)
                  varDimX = temp
                  delete(temp)
               end if
               if (dNames(k).eq.time_name) then
                  time = time_conform(varDimX)
                  time = coord_list("time",time)
                  varData!k = "time"
                  varData&time = time
                  delete(time)
               end if
               if (dNames(k).eq.lev_name) then
                  plev = coord_list("plev",varDimX)
                  varData!k = "plev"
                  varData&plev = plev
                  delete(plev)
               end if
               if (dNames(k).eq.lat_name) then
                  lat = coord_list("lat",varDimX)
                  varData!k = "lat"
                  varData&lat = lat
                  delete(lat)
               end if
               if (dNames(k).eq.lon_name) then
                  lon = coord_list("lon",varDimX)
                  varData!k = "lon"
                  varData&lon = lon
                  delete(lon)
               end if
               delete(varDimX)
            end do
            delete(varData1)
            delete(dNames)

            ; Computes the zonal average of a variable [4D -> 3D data (*2*z)]
            if ((rank.eq.4).and.(isStrSubset(field,"2")).and.(isStrSubset(field,"z"))) then
               temp = dim_avg_Wrap(varData)
               delete(varData)
               varData = temp
               delete(temp)
               rankN = 3
            else
               rankN = rank
            end if

            ;=========================================================
            ; write to output file, fout
            ;=========================================================
            if first_write then

               ; create global attributes of the file
               fAtt = True                         ; assign file attributes
               fAtt@fields = att_field
               global_attribute ( MODEL,references,contact,REF,exp_id,history,comment,source,fAtt )
               fileattdef( fout, fAtt )      ; copy file attributes

               ;fout->time = time

               ; set var_name attributes
               varData = var_list(var_name,varData)
               cell_methods = ""
               if isStrSubset(field,"z") then
                  cell_methods = "lon:mean (zonal mean), "
               end if
               if isStrSubset(field,"D") then
                  cell_methods = cell_methods + "time:mean (of day)"
               end if
               if isStrSubset(field,"M") then
                  cell_methods = cell_methods + "time:mean (of calendar months)"
               end if
               if isStrSubset(field,"I") then
                  cell_methods = cell_methods + "time:istantaneous"
               end if
               if isStrSubset(field,"A") then
                 cell_methods = cell_methods + "time:mean (of year)"
               end if
               varData@cell_methods = cell_methods
               fout->$var_name$ = varData

               if (sum(where(vNames.eq."hyam",1,0)).gt.0) then
                  fout->hyam = fin->hyam
               end if
               if (sum(where(vNames.eq."hybm",1,0)).gt.0) then
                  fout->hybm = fin->hybm
               end if
               if (sum(where(vNames.eq."hyai",1,0)).gt.0) then
                  fout->hyai = fin->hyai
               end if
               if (sum(where(vNames.eq."hybi",1,0)).gt.0) then
                  fout->hybi = fin->hybi
               end if
               if (sum(where(vNames.eq."p0",1,0)).gt.0) then
                  fout->p0 = fin->p0
               end if

               first_write = False

            else

               end_point = file_pointer+dimsizes(ind_t)-1

               fout->time(file_pointer:end_point)=(/varData&time/)

               if (rankN.eq.4) then
                  fout->$var_name$(file_pointer:end_point,:,:,:) = (/varData/)
               end if
               if (rankN.eq.3) then
                  fout->$var_name$(file_pointer:end_point,:,:) = (/varData/)
               end if
               if (rankN.eq.2) then
                  fout->$var_name$(file_pointer:end_point,:) = (/varData/)
               end if
               if (rankN.eq.1) then
                  fout->$var_name$(file_pointer:end_point) = (/varData/)
               end if
            end if
            file_pointer = file_pointer + dimsizes(ind_t)
            delete(varData)
         else
            delete(varData0)
         end if

         delete(ind_t)

      end do  ; loop (j - blocks)

      delete(fin)
      delete(vNames)
      delete(i_st)
      delete(i_ed)

   end do  ; loop (i - files)

   delete(fils)
   delete(fout)

; if first_write = True, then there is no data within the specified date range in the input file(s).
; so delete output file again and allow cf_convert.py to return 1 (=error).
   if first_write then
      system("/bin/rm " +fn_out )
      print("[attribute.ncl] FATAL: no data was within the specified date range. please check your setting!")
   end if

end
